#!/bin/bash
#
# Author: E. Auerbach, CMRR, 2013-6
#
# /opt/med/config/scripts/med.rc.setup/eja_create_rawdata_fs.sh
#
# Script to initialize eja rawdata drive on supported systems (with free drive
# slots). The rawdata drive should be physically located in the bottom-most
# slot.
# VE11 version (ext4).
#
# We are controlled by the SYSTEM_TYPE also known as Chassis Asset Tag
# which tells us which hardware platform we have.
#
# Type  Support  Description
# ----  -------  ---------------------------------------------------------------
# n/a      No     VB13 Lowend or Highend
# AA       No     VB15 standard Step 3
# AB       No     VB17 standard Step 4,  8 channels, 1 system + 1 "RAID" disk
# AC       No     VB17 standard Step 4, 18 channels, 1 system + 2 RAID disks
# AD       No     VB17 standard Step 4, 32 channels, 1 system + 3 RAID disks
#                  *** eja: my MRIR++ also uses this SYSTEM_TYPE value, but can
#                          have subtypes:
#                      EJA_SYSTEM_TYPE:
#                          EJA_MkI: 32 channels, 1 system + 3 RAID disks
# BA       No     VC11 MARS 1 disk
# CA       No     VC12 MARS 1 SATA disk
# CB      Yes     VD11 MARS2,  64 channels: 1 system + 4 RAID disks
# CC      Yes     VD11 MARS2,  64 channels: 1 system + 4 RAID disks
# DA      Yes     VD11 MARS2,  48 channels: 1 system + 3 RAID disks
# DB      Yes     VD11 MARS2,  64 channels: 1 system + 4 RAID disks
# DC      Yes     VD11 MARS2, 128 channels: 1 system + 6 RAID disks
# DD      Yes     VD12 MARS2,  24 channels: 1 system + 2 RAID disks
# DE      Yes     VE11 MARS2,  48 channels: 1 system + 2 RAID disks
# DF      Yes     VE11 MARS2,  64 channels: 1 system + 3 RAID disks
# DG      Yes     VE11 MARS2, 128 channels: 1 system + 5 RAID disks
# DH      Yes     VE15 MARS2,  24 channels: 1 system + 2 RAID disks
# DI       No     BR StepD.2,  32 channels: 1 system + 7 RAID disks
# VM       ?      virtual machine

# be sure we know the system type (it usually isn't exported on VE11)
if [ -e /etc/system.type ]
then
    . /etc/system.type
fi

# raw data disk should always be in bottom-most hot-swap slot (#8), or at least
# after the last RAID disk.
if [ "$SYSTEM_TYPE" = "" ]
then
    echo "eja_create_rawdata_fs.sh: SYSTEM_TYPE environment variable not set/exported! Aborting..."
    exit 1
elif [ "$SYSTEM_TYPE" = "DD" ] || [ "$SYSTEM_TYPE" = "DE" ] || [ "$SYSTEM_TYPE" = "DH" ]
then
    # 1 system + 2 RAID disks
    EJA_RAWDATA_DISK="/dev/sdd"
elif [ "$SYSTEM_TYPE" = "DA" ] || [ "$SYSTEM_TYPE" = "DF" ]
then
    # 1 system + 3 RAID disks
    EJA_RAWDATA_DISK="/dev/sde"
elif [ "$SYSTEM_TYPE" = "CB" ] || [ "$SYSTEM_TYPE" = "CC" ] || [ "$SYSTEM_TYPE" = "DB" ]
then
    # 1 system + 4 RAID disks
    EJA_RAWDATA_DISK="/dev/sdf"
elif [ "$SYSTEM_TYPE" = "DG" ]
then
    # 1 system + 5 RAID disks
    EJA_RAWDATA_DISK="/dev/sdg"
elif [ "$SYSTEM_TYPE" = "DC" ]
then
    # 1 system + 6 RAID disks
    EJA_RAWDATA_DISK="/dev/sdh"
else
    echo "eja_create_rawdata_fs.sh: Unknown or unsupported system type ${SYSTEM_TYPE}! Aborting..."
    exit 1
fi

# be sure the necessary tools are available
check_available()
{
    if [ -z `which $1` ]
    then
        echo "eja_create_rawdata_fs.sh: $1 not found! Aborting..."
        exit 1
    fi
}

check_available "sed";
check_available "grep";
check_available "sfdisk";
check_available "mkfs.ext4";

# partition raw data disk if it exists
if [ $EJA_RAWDATA_DISK ]
then
    if [ -z "$EJA_RAWDATA_DISK" ]
    then
        echo "eja_create_rawdata_fs.sh: Couldn't find raw data disk at $EJA_RAWDATA_DISK!! Aborting..."
        exit 1;
    fi

    RAWDATA_DISK_SIZE=`sfdisk -l $EJA_RAWDATA_DISK | grep "Disk.*cylinders" | sed -e "s/Disk.*: \([0-9]\+\).*/\1/g"`
    echo "eja_create_rawdata_fs.sh: Found raw data disk with $RAWDATA_DISK_SIZE cylinders at device $EJA_RAWDATA_DISK"
    echo "eja_create_rawdata_fs.sh: Partitioning raw data disk..."
sfdisk $EJA_RAWDATA_DISK << EOF
0,$RAWDATA_DISK_SIZE
;
;
;
EOF
    if [ $? -ne 0 ]
    then
        echo "eja_create_rawdata_fs.sh: Partitioning raw data disk FAILED";
        exit 1;
    fi

    echo "eja_create_rawdata_fs.sh: Creating raw data filesystem..."
    mkfs.ext4 -q ${EJA_RAWDATA_DISK}1 || exit 1;
fi
