package org.nrg.xnat.restlet.extensions;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.jackson.map.ObjectMapper;
import org.nrg.config.entities.Configuration;
import org.nrg.config.exceptions.ConfigServiceException;
import org.nrg.config.services.ConfigService;
import org.nrg.xdat.XDAT;
import org.nrg.xdat.om.XnatProjectdata;
import org.nrg.xft.exception.DBPoolException;
import org.nrg.xnat.protocol.Protocol;
import org.nrg.xnat.restlet.XnatRestlet;
import org.nrg.xnat.restlet.resources.SecureResource;
import org.nrg.xnat.restlet.util.FileWriterWrapperI;
import org.restlet.Context;
import org.restlet.data.MediaType;
import org.restlet.data.Request;
import org.restlet.data.Response;
import org.restlet.data.Status;
import org.restlet.resource.Representation;
import org.restlet.resource.Variant;


// validates a protocol (won't let you upload a bad one. If someone uploaded a bad one directly using the config 
// service, it won't let you download it.) handles get, put and delete

@XnatRestlet({"/projects/{PROJECT_ID}/protocol"})
public class ProjectProtocolResource  extends SecureResource {

	private static final Log _log = LogFactory.getLog(ProjectProtocolResource.class);
	private final ObjectMapper _mapper = new ObjectMapper();
	private final ConfigService configService = XDAT.getConfigService();
	private final String _projectId;
	private XnatProjectdata _project;
	//private Protocol theProtocol = null;
	//private String protocolJson = null;
	Callable<Long> getProjectId = null;
	
	private static final String EMPTY_RESULT_SET = "{}";
	private static final String TOOL_NAME = "protocols";
	private static final String PATH = "protocol";
	
    public ProjectProtocolResource(Context context, Request request, Response response) throws SQLException, DBPoolException {
        super(context, request, response);
        setModifiable(true);
        getVariants().add(new Variant(MediaType.APPLICATION_JSON));

        final Map<String, Object> attributes = request.getAttributes();
        _projectId = attributes.containsKey("PROJECT_ID") ? (String) attributes.get("PROJECT_ID") : null;


        final boolean hasProjectID = !StringUtils.isBlank(_projectId);
        if (!hasProjectID) {
            response.setStatus(Status.CLIENT_ERROR_PRECONDITION_FAILED, "You must specify a project ID for this service call.");        
            return;
        }

        _log.info(String.format("Found service call for project %s.", _projectId));

        _project = XnatProjectdata.getProjectByIDorAlias(_projectId, user, false);

        if (_project == null) {
            response.setStatus(Status.CLIENT_ERROR_NOT_FOUND, "The project ID " + _projectId + " does not result in a valid project.");
            _log.warn("Found no corresponding project for requested project ID: " + _projectId + ", requested by user: " + user.getUsername());
            return;
        }
        
        //used for config service
		getProjectId = new Callable<Long>() { public Long call() { return new Long((Integer)_project.getItem().getProps().get("projectdata_info"));}};
        
    }

	@Override
	public boolean allowDelete() {
		return true;
	}
	@Override
	public void handleDelete(){
		if(configService.getConfigContents(TOOL_NAME, PATH, getProjectId)!=null){
			configService.disable(user.getUsername(), "deleting via protocol rest api", TOOL_NAME, PATH, getProjectId);
		}
	}
	
	@Override
	public boolean allowPut() {
		return true;
	}
	
	@Override
	public void handlePut(){
		try {
			Representation entity = null;
			FileWriterWrapperI fw = null;
			
			entity = this.getRequest().getEntity();
			List<FileWriterWrapperI> fws = this.getFileWritersAndLoadParams(entity);
			if (fws.size() == 0) {
				this.getResponse().setStatus(Status.CLIENT_ERROR_BAD_REQUEST, "Unable to identify upload format.");
				return;
			}

			if(fws.size()>1){
				this.getResponse().setStatus(Status.CLIENT_ERROR_BAD_REQUEST, "Importer is limited to one uploaded resource at a time.");
				return;
			}
			fw = fws.get(0);

			//read the input stream into a string buffer.
			final InputStream is = fw.getInputStream();
			BufferedReader reader = new BufferedReader(new InputStreamReader(is));
		    StringBuilder sb = new StringBuilder();
		    String line = null;
		    while ((line = reader.readLine()) != null) {
		      sb.append(line + "\n");
		    }
		    is.close();
		    String contents = sb.toString();
		    
		    //validate the contents (the protocol string)
		    ObjectMapper mapper = new ObjectMapper();
			try {
				Protocol theProtocol = mapper.readValue(contents, Protocol.class);	
				String validateString = theProtocol.validate();
				if(validateString != null){
					logger.error("the protocol is invalid: " + validateString);
					this.getResponse().setStatus(Status.SERVER_ERROR_INTERNAL,"The protocol found for requested project ID: " + _projectId + " cannot be parsed. " + validateString);
					return;
				}
			} catch (Exception e){
				this.getResponse().setStatus(Status.SERVER_ERROR_INTERNAL,"The protocol found for requested project ID: " + _projectId + " cannot be parsed. " + e.getMessage());
				return;
			}
		    
			//TODO: at this point we want to run it through further validation.
		    
		    //if there is a previous configuration check to see if its contents equals the new contents, if so, just return success.
			//do not update the configuration for puts are idempotent
		    Configuration prevConfig = configService.getConfig(TOOL_NAME, PATH, getProjectId);
			if(prevConfig != null && contents.equals(prevConfig.getContents())) {
				if(StringUtils.equals(Configuration.DISABLED_STRING, prevConfig.getStatus())){
					configService.enable(user.getUsername(), "enabled via protocol rest api", TOOL_NAME, PATH, getProjectId);
				}
				getResponse().setStatus(Status.SUCCESS_OK);
			} else {
				//save/update the configuration
				if(prevConfig != null && StringUtils.equals(Configuration.DISABLED_STRING, prevConfig.getStatus())){
					configService.enable(user.getUsername(), "enabled via protocol rest api", TOOL_NAME, PATH, getProjectId);
				}
				
				configService.replaceConfig(user.getUsername(), "uploaded via protocol rest aip", TOOL_NAME, PATH, contents, getProjectId);
				getResponse().setStatus(Status.SUCCESS_CREATED);
			}
		}catch (ConfigServiceException e) {
			this.getResponse().setStatus(Status.SERVER_ERROR_INTERNAL, e.getMessage());
		} catch (Exception e){
			this.getResponse().setStatus(Status.SERVER_ERROR_INTERNAL, e.getMessage());
		}
	}
	
	
    @Override
    public void handleGet() {
    	
    	Configuration config = configService.getConfig(TOOL_NAME, PATH, getProjectId);
    	
    	if(config == null || StringUtils.equalsIgnoreCase(Configuration.DISABLED_STRING, config.getStatus() ) ){
    		getResponse().setEntity(EMPTY_RESULT_SET, MediaType.APPLICATION_JSON);
    		return;
    	}

    	String protocolJson = config.getContents();
		    	
        if (protocolJson != null) {
            try {            	
            	//validate it
        		ObjectMapper mapper = new ObjectMapper();
    			mapper.readValue(protocolJson, Protocol.class);				
                getResponse().setEntity(protocolJson, MediaType.APPLICATION_JSON);
            } catch (Exception exception) {
                respondToException(exception, Status.SERVER_ERROR_INTERNAL);
            }
        } else {
            getResponse().setEntity(EMPTY_RESULT_SET, MediaType.APPLICATION_JSON);
        }
    }
}
